<?php

namespace App\Http\Controllers;

use App\Models\Signature;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class SignatureController extends Controller
{
    // Ambil semua data signature
    public function index()
    {
        $signatures = Signature::all();

        foreach ($signatures as $signature) {
            if (!empty($signature->signature_img) && Storage::exists($signature->signature_img)) {
                $path = $signature->signature_img;
                $base64 = base64_encode(Storage::get($path));
                $signature->signature_img = 'data:image/png;base64,' . $base64;
            } else {
                $signature->signature_img = null;
            }
        }

        $response = [
            'success' => true,
            'data' => $signatures
        ];

        return response()->json($response, 200);
    }

    // Simpan data baru
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'position' => 'nullable|string|max:255',
            'signature_img' => 'nullable|file|mimes:png,jpg,jpeg',
        ]);

        $path = null;

        if ($request->hasFile('signature_img')) {
            $randomDigits = Str::random(5);
            $fileName = 'signature_img_' . $randomDigits . '.' . $request->file('signature_img')->getClientOriginalExtension();
            $path = Storage::putFileAs('signatures', $request->file('signature_img'), $fileName);
            $request->merge(['signature_img' => $path]);
        }

        $signature = new Signature();
        $signature->name = $request->name;
        $signature->position = $request->position;
        $signature->signature_img = $path;
        $signature->save();

        $response = [
            'success' => true,
            'message' => 'Data signature berhasil disimpan',
            'data' => $signature
        ];

        return response()->json($response, 201);
    }

    // Ambil data berdasarkan ID
    public function show($id)
    {
        $signature = Signature::find($id);

        if (!$signature) {
            return response()->json(['message' => 'Data tidak ditemukan'], 404);
        }

        $image_data = null;
        if (isset($signature->signature_img)) {
            $path = $signature->signature_img;
            $base64 = base64_encode(Storage::get($path));
            $image_data = 'data:image/png;base64,' . $base64;
        }

        $response = [
            'signature' => $signature,
            'signature_img' => $image_data
        ];

        return response()->json($response);
    }

    // Update data berdasarkan ID
    public function update(Request $request)
    {
        $signature = Signature::find($request->id);

        if (!$signature) {
            return response()->json(['message' => 'Data tidak ditemukan'], 404);
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'position' => 'nullable|string|max:255',
            'signature_img' => 'nullable|file|mimes:png,jpg,jpeg',
        ]);

        if ($request->hasFile('signature_img')) {
            // Hapus gambar lama
            if (Storage::exists($signature->signature_img)) {
                Storage::delete($signature->signature_img);
            }

            // Simpan gambar baru dengan nama unik
            $randomDigits = Str::random(5);
            $fileName = 'signature_img_' . $randomDigits . '.' . $request->file('signature_img')->getClientOriginalExtension();
            $path = Storage::putFileAs('signatures', $request->file('signature_img'), $fileName);
            $request->merge(['signature_img' => $path]);
            $signature->signature_img = $path;
        }

        $signature->name = $request->name;
        $signature->position = $request->position;
        $signature->save();

        $response = [
            'success' => true,
            'message' => 'Data signature berhasil diperbarui',
            'data' => $signature
        ];

        return response()->json($response);
    }

    // Hapus data berdasarkan ID
    public function destroy($id)
    {
        $signature = Signature::find($id);

        if (!$signature) {
            return response()->json(['message' => 'Data tidak ditemukan'], 404);
        }

        // Hapus gambar dari penyimpanan
        Storage::delete($signature->signature_img);

        $signature->delete();

        $response = [
            'success' => true,
            'message' => 'Data signature berhasil dihapus'
        ];

        return response()->json($response);
    }
}
